#include "eaiproxy.h"
#include "eaiexecutor.h"
#include "eappaiprompt.h"
#include "eaicallbck.h"
#include "epptwebview.h"
#include "eposterwebview.h"
#include "eparserdocument.h"
#include "localmodelserver.h"
#include "easyncworker.h"
#include "esystemcontext.h"

#include <QScopedPointer>
#include <QApplication>
#include <QClipboard>
#include <QDBusMessage>
#include <QDBusPendingCall>
#include <QLoggingCategory>

Q_DECLARE_LOGGING_CATEGORY(logAIGUI)

using namespace uos_ai;

EAiProxy::EAiProxy(QObject *parent)
    : QObject{parent}
{
    connect(EAiExec(), &EAiExecutor::llmAccountLstChanged,
            this, &EAiProxy::llmAccountLstChanged);

    //Init audio singals
    connect(EAiExec(), &EAiExecutor::audioASRInComing,
            this, &EAiProxy::sigAudioASRStream);
    connect(EAiExec(), &EAiExecutor::audioASRError,
            this, &EAiProxy::sigAudioASRError);
    connect(EAiExec(), &EAiExecutor::playTTSError,
            this, &EAiProxy::sigPlayTTSError);
    connect(EAiExec(), &EAiExecutor::playTTSFinished,
            this, &EAiProxy::sigPlayTTSFinished);
    connect(EAiExec(), &EAiExecutor::audioOutputDeviceChanged,
            this, &EAiProxy::sigAudioOutputDevChanged);
    connect(EAiExec(), &EAiExecutor::audioInputDeviceChange,
            this, &EAiProxy::sigAudioInputDevChange);
    connect(EAiExec(), &EAiExecutor::audioSampleLevel,
            this, &EAiProxy::sigAudioSampleLevel);

    //Init TTP signals
    connect(EAiExec(), &EAiExecutor::chatConversationType,
            this, &EAiProxy::sigChatConversationType);

    connect(EAiExec(), &EAiExecutor::textToPictureFinish,
            this, &EAiProxy::sigText2PicFinish);
    connect(EAiExec(), &EAiExecutor::pptCreateSuccess,
            this, &EAiProxy::sigPPTCreateSuccess);
    connect(EAiExec(), &EAiExecutor::pptChangeSuccess,
            this, &EAiProxy::sigPPTChangeSuccess);
    connect(EAiExec(), &EAiExecutor::posterCreateSuccess,
            this, &EAiProxy::sigPosterCreateSuccess);
    //Try to stop all requst from this proxy when it's
    //going to be destructed
    connect(this, &EAiProxy::destroyed, this, [this]() {
        EAiExec()->clearAiRequest(this);
    });

    //Network signal
    connect(EAiExec(), &EAiExecutor::netStateChanged,
            this, &EAiProxy::sigNetStateChanged);

    connect(DGuiApplicationHelper::instance(), &DGuiApplicationHelper::themeTypeChanged,
    this, [this](DGuiApplicationHelper::ColorType themeType) {
        //Check if the theme's changed.
        if (m_themeType != themeType) {
            m_themeType = themeType;
            emit sigThemeChanged(m_themeType);
        }

        //Check if the active color is changed.
        QString activeColor = DGuiApplicationHelper::instance()->applicationPalette()
                              .color(DPalette::Normal, DPalette::Highlight)
                              .name(QColor::HexRgb);
        if (m_activeColor != activeColor) {
            m_activeColor = activeColor;
            emit sigActiveColorChanged(m_activeColor);
        }
    });
    connect(QApplication::instance(), SIGNAL(fontChanged(const QFont &)), this, SLOT(onUpdateSystemFont(const QFont &)));

    m_themeType = DGuiApplicationHelper::instance()->themeType();

    m_activeColor = DGuiApplicationHelper::instance()->applicationPalette()
                    .color(DPalette::Normal, DPalette::Highlight)
                    .name(QColor::HexRgb);

    //Load toast message
    m_aiToastMessage.append("");
    m_aiToastMessage.append(QCoreApplication::translate("AiToastMessage", "Chat history cleared"));
    m_aiToastMessage.append(QCoreApplication::translate("AiToastMessage", "Copied successfully"));

    m_audioRecCounter.reset(new QTimer(this));
    connect(m_audioRecCounter.get(), &QTimer::timeout, this, [this]() {
        m_audioLenLimit--;

        if (m_audioLenLimit <= 10) {
            emit sigAudioCountDown(m_audioLenLimit);

            //Stop count at zero
            if (m_audioLenLimit == 0) {
                m_audioRecCounter->stop();
                qCDebug(logAIGUI) << "Audio recording time limit reached";
            }
        }
    });

    connect(EAiExec(), &EAiExecutor::knowledgeBaseExistChanged, this, &EAiProxy::sigKnowledgeBaseStatusChanged);
    connect(EAiExec(), &EAiExecutor::knowledgeBaseFAQGenFinished, this, &EAiProxy::sigKnowledgeBaseFAQGenFinished);

    connect(EAiExec(), &EAiExecutor::localLLMStatusChanged, this, &EAiProxy::sigLocalLLMStatusChanged);
    connect(EAiExec(), &EAiExecutor::embeddingPluginsStatusChanged, this, &EAiProxy::sigEmbeddingPluginsStatusChanged);

    // 拖拽文档解析
    connect(EAiExec(), &EAiExecutor::docSummaryParsingStart, this, &EAiProxy::sigDocSummaryParsingStart);
    connect(EAiExec(), &EAiExecutor::docDragInViewParserResult, this, &EAiProxy::sigDocSummaryParserResult);
    // API: openFile() -> Result
    connect(EAiExec(), &EAiExecutor::openFileFromPathResult, this, &EAiProxy::sigOpenFileFromPathResult);

    connect(EAiExec(), &EAiExecutor::sigAppendWordWizardConv, this, &EAiProxy::sigAppendWordWizardConv);

    connect(EAiExec(), &EAiExecutor::previewReference, this, &EAiProxy::sigPreviewReference);

    connect(EAiExec(), &EAiExecutor::sigOverrideQues, this, &EAiProxy::sigOverrideQues);
    connect(EAiExec(), &EAiExecutor::sigAssistantListChanged, this, &EAiProxy::sigAssistantListChanged);

    connect(EAiExec(), &EAiExecutor::sigGetNewHistoryList, this, &EAiProxy::sigGetNewHistoryList);
    connect(EAiExec(), &EAiExecutor::sigHideHistoryList, this, &EAiProxy::sigHideHistoryList);

    connect(EAiExec(), &EAiExecutor::sigToShowPromptWindow, this, &EAiProxy::sigToShowPromptWindow);

    connect(EAiExec(), &EAiExecutor::sigMainContentBackgroundColor, this, [this](QString color){
        m_backgroundColor = color;
        emit sigMainContentBackgroundColor(color);
    });

    connect(EAiExec(), &EAiExecutor::sigInputFocus, this, &EAiProxy::sigInputFocus);

    initAsyncWorkerConn();
}

void EAiProxy::onUpdateSystemFont(const QFont &font)
{
    QFontInfo fontInfo(font);
    setFontInfo(fontInfo.family(), fontInfo.pixelSize());
    emit sigFontChanged(fontInfo.family(), fontInfo.pixelSize());
}

bool EAiProxy::isKnowledgeBaseExist()
{
    return EAiExec()->isKnowledgeBaseExist();
}

bool EAiProxy::isEmbeddingPluginsExist()
{
    return EAiExec()->isEmbeddingPluginsExist();
}

void EAiProxy::configureKnowledgeBase()
{
    return EAiExec()->launchLLMConfigWindow(false);
}

void EAiProxy::installEmbeddingPlugins()
{
    return EAiExec()->openInstallWidget(PLUGINSNAME);
}

void EAiProxy::onDocSummarySelect()
{
    EAiExec()->documentSummarySelect();
}

void EAiProxy::onDocSummaryParsing(const QString &docPath)
{
    EAiExec()->documentSummaryParsing(docPath);
}

void EAiProxy::openFile(const QString &filePath)
{
    EAiExec()->openFile(filePath);
}

void EAiProxy::openUrl(const QString &url)
{
    EAiExec()->openUrl(url);
}

void EAiProxy::appendWordWizardConv(int type)
{
    EAiExec()->appendWordWizardConv(type);
}

void EAiProxy::webViewLoadFinished()
{
    EAiExec()->webViewLoadFinished();
}

void EAiProxy::previewRefDoc(const QString &docPath, const QStringList &docContents)
{
    EAiExec()->previewRefDoc(docPath, docContents);
}

QString EAiProxy::getInstList()
{
    return EAiExec()->getInstList();
}

void EAiProxy::sendPPTOutline(const QString &content)
{
    EPPTWebView *pptweb = ESystemContext::createWebWindow<EPPTWebView>(content, "2", "0");
    connect(pptweb, &EPPTWebView::sigPPTCreated,
            EAiExec(), &EAiExecutor::onPPTCreated);
    connect(pptweb, &EPPTWebView::sigPPTChanged,
            EAiExec(), &EAiExecutor::onPPTChanged);
}

void EAiProxy::editPPT(const QString &content)
{
    EPPTWebView *pptweb = ESystemContext::createWebWindow<EPPTWebView>(content, "1", "0");
    connect(pptweb, &EPPTWebView::sigPPTCreated,
            EAiExec(), &EAiExecutor::onPPTCreated);
    connect(pptweb, &EPPTWebView::sigPPTChanged,
            EAiExec(), &EAiExecutor::onPPTChanged);
}

void EAiProxy::downloadPPT(const QString &id)
{
    EPPTWebView *pptweb = ESystemContext::createWebWindow<EPPTWebView>("", "1", id);
    connect(pptweb, &EPPTWebView::sigPPTCreated,
            EAiExec(), &EAiExecutor::onPPTCreated);
    connect(pptweb, &EPPTWebView::sigPPTChanged,
            EAiExec(), &EAiExecutor::onPPTChanged);
}

void EAiProxy::editPoster(const QString &content)
{
    QRegularExpression regex("ID：(\\d+)");
    QRegularExpressionMatch match = regex.match(content);

    if (match.hasMatch()) {
        // 提取ID
        QString id = match.captured(1);
        EPosterWebView *posterweb = ESystemContext::createWebWindow<EPosterWebView>(EPosterWebView::WebSitePage::ModifyParameters, id);
        connect(posterweb, &EPosterWebView::sigPosterCreated, EAiExec(), &EAiExecutor::onPosterCreated);
    } else {
        qCWarning(logAIGUI) << "not found id.";
    }
}

void EAiProxy::genePoster(const QString &content)
{
    QRegularExpression regex("ID：(\\d+)");
    QRegularExpressionMatch match = regex.match(content);

    if (match.hasMatch()) {
        // 提取ID
        QString id = match.captured(1);
        EPosterWebView *posterweb = ESystemContext::createWebWindow<EPosterWebView>(EPosterWebView::WebSitePage::GenerateWorks, id);
        connect(posterweb, &EPosterWebView::sigPosterCreated, EAiExec(), &EAiExecutor::onPosterCreated);
    } else {
        qCWarning(logAIGUI) << "not found id.";
    }
}

void EAiProxy::downloadPoster(const QString &id)
{
    EPosterWebView *posterweb = ESystemContext::createWebWindow<EPosterWebView>(EPosterWebView::WebSitePage::ModifyWork, id);
    connect(posterweb, &EPosterWebView::sigPosterCreated, EAiExec(), &EAiExecutor::onPosterCreated);
}

bool EAiProxy::isAgentSupported()
{
#ifdef ENABLE_AGENT_PLUGIN
    return true;
#else
    return false;
#endif
}

void EAiProxy::openAppstore(const QString &id)
{
    if (id == "agent") {
        auto con = QDBusConnection::sessionBus();
        QDBusMessage msg = QDBusMessage::createMethodCall("com.home.appstore.client", "/com/home/appstore/client",
                                       "com.home.appstore.client", "openBusinessUri");

        QVariantList args;
        args << QVariant::fromValue(QString("tab/topAiAgentApp"));
        msg.setArguments(args);

        con.asyncCall(msg);
    }
}

void EAiProxy::onAsyncWorkerFinished(int type)
{
    emit EAsync()->sigWorkerFinished(type);
}

void EAiProxy::rateAnwser(const int questionIdx, const int answerIdx, int rate, const QString &extJson)
{
    EAiExec()->rateAnwser(questionIdx, answerIdx, rate, extJson);
}

void EAiProxy::setTitleBarMaskStatus(bool status)
{
    EAiExec()->setTitleBarMaskStatus(status);
}

void EAiProxy::showWarningDialog(const QString assistantId, const QString conversationId, const QString msg, bool isDelete, bool isLlmDelete)
{
    EAiExec()->showWarningDialog(assistantId, conversationId, msg, isDelete, isLlmDelete);
}

void EAiProxy::updateUpdatePromptDB(bool isClicked)
{
    EAiExec()->updateUpdatePromptDB(isClicked);
}

QString EAiProxy::getNotifierName(AiAction act, int mode)
{
    Q_UNUSED(act);

    QString actionName("");

    actionName = (Mode::CacheMode == mode)
                 ? GET_NOTIFIER_NAME(AiReply)
                 : GET_SNOTIFIER_NAME(AiReply);

    return actionName;
}

void EAiProxy::initAsyncWorkerConn()
{
    connect(EAsync(), &EAsyncWorker::sigAsyncWorker, this, &EAiProxy::sigAsyncWorker);
}

QString EAiProxy::sendAiRequest(const QString &llmId, int llmType, int act, const QString &param, int mode)
{
    QString reqId("");

    if ((act > None && act < MaxAiAction)
            && (mode > Mode::None && mode < Mode::MaxMode)
            && (!param.isEmpty())
            /*&& (!llmId.isEmpty())*/ //Allow no account request
       ) {
        QSharedPointer<EAiPrompt> aiPrompt;
        QSharedPointer<EAiCallback> aiCallback;

        aiCallback.reset(new EAiCacheCallback(this));
        aiCallback->setDataMode(Mode(mode));

        //act param isn't used
        aiCallback->setNotifier(getNotifierName(None, mode));

        switch (act) {
        case Conversation:
            aiPrompt.reset(new EConversationPrompt(param));
            aiCallback->setOp(Conversation);
            break;
        case DocumentSummary:
            aiPrompt.reset(new EAiDocSummaryPrompt(param, QString("")));
            aiCallback->setOp(DocumentSummary);
            break;
        default:
            qCWarning(logAIGUI) << "Unimplemented operation:" << act;
            break;
        }

        //Set ai model type
        if (!aiPrompt.isNull()) {
            aiPrompt->setLLM(llmType);
            reqId = EAiExec()->sendAiRequst(llmId, aiPrompt, aiCallback);
            qCDebug(logAIGUI) << "AI request sent - ID:" << reqId << "Type:" << act << "Mode:" << mode;
        }
    } else {
        qCWarning(logAIGUI) << "Invalid request parameters - llmId:" << llmId 
                           << "act:" << act 
                           << "mode:" << mode 
                           << "param empty:" << param.isEmpty();
    }

    return reqId;
}

QString EAiProxy::sendRequest(const QString &llmId, const QString &chatChunkData)
{
    uos_ai::ChatChunk chatChunk;
    uos_ai::ChatChunk::json2ChatChunk(chatChunkData.toUtf8(), chatChunk);
    EAiExec()->sendRequest(llmId, chatChunk, this, getNotifierName(None, Mode::StreamMode));

    QJsonObject chatChunkObj;
    uos_ai::ChatChunk::chatChunk2Json(chatChunkObj, chatChunk);
    return QJsonDocument(chatChunkObj).toJson(QJsonDocument::Compact);
}

void EAiProxy::cancelAiRequest(const QString &id)
{
    qCDebug(logAIGUI) << "Cancelling AI request:" << id;
    EAiExec()->cancelAiRequst(id);
}

QString EAiProxy::currentLLMAccountId()
{
    return EAiExec()->currentLLMAccountId();
}

QString EAiProxy::queryLLMAccountList()
{
    return EAiExec()->queryLLMAccountList();
}

bool EAiProxy::setCurrentLLMAccountId(const QString &id)
{
    return EAiExec()->setCurrentLLMAccountId(id);
}

QString EAiProxy::currentAssistantId()
{
    return EAiExec()->currentAssistantId();
}

QString EAiProxy::queryAssistantList()
{
    return EAiExec()->queryAssistantList();
}

bool EAiProxy::setCurrentAssistantId(const QString &id)
{
    return EAiExec()->setCurrentAssistantId(id);
}

QString EAiProxy::getAiFAQ()
{
    return EAiExec()->getRandomAiFAQ();
}

QString EAiProxy::getAiFAQByFunction(int type, const QString &function)
{
    return EAiExec()->getRandomAiFAQByFunction(type, function);
}

QString EAiProxy::getAssistantFunctions(int type)
{
    return EAiExec()->getAssistantFunctions(type);
}

QString EAiProxy::getFunctionTemplate(int type, const QString &function, const QString &contain)
{
    return EAiExec()->getFunctionTemplate(type , function, contain);
}

void EAiProxy::launchLLMConfigWindow(bool showAddllmPage)
{
    QString locateTitle = "";
    if (!showAddllmPage)
        locateTitle = tr("Model Configuration");

    return EAiExec()->launchLLMConfigWindow(showAddllmPage, false, false, locateTitle);
}

void EAiProxy::launchKnowledgeBaseConfigWindow()
{
    return EAiExec()->launchKnowledgeBaseConfigWindow();
}

void EAiProxy::launchAboutWindow()
{
    return EAiExec()->launchAboutWindow();
}

void EAiProxy::setFontInfo(const QString &fontFamily, int pixelSize)
{
    m_fontFamily = fontFamily;
    m_fontPixelSize = pixelSize;
}

QString EAiProxy::fontInfo()
{
    return m_fontFamily + "#" + QString::number(m_fontPixelSize);
}

void EAiProxy::setWindowMode(bool isWindowMode)
{
    m_isWindowMode = isWindowMode;
}

bool EAiProxy::isWindowMode()
{
    return m_isWindowMode;
}


void EAiProxy::showToast(int type)
{
    auto index = AiToast(type);

    if (index > 0 && type < m_aiToastMessage.size()) {
        EAiExec()->showToast(m_aiToastMessage[index]);
    }
}

void EAiProxy::closeChatWindow()
{
    EAiExec()->closeChatWindow();
}

bool EAiProxy::isAudioInputAvailable()
{
    return EAiExec()->isAudioInputAvailable();
}

bool EAiProxy::isAudioOutputAvailable()
{
    return EAiExec()->isAudioOutputAvailable();
}

bool EAiProxy::startRecorder(int mode)
{
    qCDebug(logAIGUI) << "Starting audio recorder - mode:" << mode;
    m_audioLenLimit = AUDIO_LENGTH;

    //Stop time anyway.
    m_audioRecCounter->stop();
    m_audioRecCounter->start(1000);

    return EAiExec()->startRecorder(mode);
}

bool EAiProxy::stopRecorder()
{
    qCDebug(logAIGUI) << "Stopping audio recorder";
    m_audioRecCounter->stop();

    return EAiExec()->stopRecorder();
}

bool EAiProxy::playTextAudio(const QString &id, const QString &text, bool isEnd)
{
    return EAiExec()->playTextAudio(id, text, isEnd);
}

bool EAiProxy::stopPlayTextAudio()
{
    return EAiExec()->stopPlayTextAudio();
}

void EAiProxy::playSystemSound(int effId)
{
    return EAiExec()->playSystemSound(effId);
}

void EAiProxy::copyReplyText(const QString &reply)
{
    qCDebug(logAIGUI) << "Copying reply text to clipboard";
    if (ESystemContext::isTreeland()) {
        QTimer::singleShot(10, this, [reply]{
            QClipboard *clip = QApplication::clipboard();
            clip->setText(reply);
        });
        return;
    }

    QClipboard *clip = QApplication::clipboard();
    clip->setText(reply);
}

void EAiProxy::logAiChatRecord(const QString &reqId,
                               const QString &question,
                               const QString &anwser,
                               bool isRetry,
                               int err,
                               const QString &errorInfo,
                               int actionType,
                               const QString &llmIcon,
                               const QString &llmName,
                               const QString &docSummaryParam)
{
    qCDebug(logAIGUI) << "Logging chat record - ID:" << reqId 
                      << "Action:" << actionType 
                      << "Error:" << err;
    return EAiExec()->logChatRecord(
               reqId, question, QStringList(anwser),
               isRetry, err, errorInfo,
               actionType,
               llmIcon, llmName, docSummaryParam);
}

void EAiProxy::logAiChatRecord(const QString &reqId,
                               const QString &question,
                               const QStringList &anwser,
                               bool isRetry,
                               int err,
                               const QString &errorInfo,
                               int actionType,
                               const QString &llmIcon,
                               const QString &llmName,
                               const QString &docSummaryParam)
{
    return EAiExec()->logChatRecord(
               reqId, question, anwser,
               isRetry, err, errorInfo,
               actionType,
               llmIcon, llmName, docSummaryParam);
}

QString EAiProxy::getAiChatRecords(bool lastRec)
{
    return EAiExec()->getChatRecords(lastRec);
}

QString EAiProxy::getConversations()
{
    return EAiExec()->getConversations();
}

QString EAiProxy::createNewConversation()
{
    return EAiExec()->createNewConversation();
}

void EAiProxy::removeConversation(const QString &assistantId, const QString &conversationId)
{
    return EAiExec()->removeConversation(assistantId, conversationId);
}

QString EAiProxy::getConversationHistoryList()
{
    return EAiExec()->getConversationHistoryList();
}

bool EAiProxy::setCurrentConversationId(const QString &assistantId, const QString &conversationId)
{
    return EAiExec()->setCurrentConversationId(assistantId, conversationId);
}

QString EAiProxy::getLastConversation(const QString &assistantId)
{
    return EAiExec()->getLastConversation(assistantId);
}

void EAiProxy::clearAiChatRecords()
{
    qCDebug(logAIGUI) << "Clearing all AI chat records";
    return EAiExec()->clearChatRecords();
}

void EAiProxy::logCurrentConversations(const QString &assistantId, const QString &conversationId, const QString &assistantDisplayName, const QString &conversationsData)
{
    QVector<uos_ai::Conversations> convs;
    uos_ai::Conversations::json2Convs(convs, conversationsData.toUtf8());

    EAiExec()->logConversations(assistantId, conversationId, assistantDisplayName, convs);
}

bool EAiProxy::saveImageAs(const QString &filePath)
{
    return EAiExec()->saveImageAs(filePath);
}

bool EAiProxy::previewImage(const QString &filePath)
{
    return EAiExec()->previewImage(filePath);
}

void EAiProxy::copyImage2Clipboard(const QString &filePath)
{
    return EAiExec()->copyImg2Clipboard(filePath);
}

bool EAiProxy::isNetworkAvailable()
{
    bool available = EAiExec()->isNetworkAvailable();
    qCDebug(logAIGUI) << "Network status:" << available;
    return available;
}

QJsonObject EAiProxy::loadTranslations()
{
    static QJsonObject translations;
    if (!translations.isEmpty())
        return translations;

    translations["Go to configuration"] = tr("Go to configuration");
    translations["No account"] = tr("No account");
    translations["Input question"] = tr("Input question");
    translations["The content generated by AI is for reference only, please pay attention to the accuracy of the information."] = tr("The content generated by AI is for reference only, please pay attention to the accuracy of the information.");
    translations["Welcome to UOS AI"] = tr("Welcome to UOS AI");
    translations["Here are some of the things UOS AI can help you do"] = tr("Here are some of the things UOS AI can help you do");
    translations["Stop"] = tr("Stop");
    translations["Regenerate"] = tr("Regenerate");
    translations["Clear conversation history"] = tr("Clear conversation history");
    translations["Please connect the microphone and try again"] = tr("Please connect the microphone and try again");
    translations["Chat history cleared"] = tr("Chat history cleared");
    translations["No account"] = tr("No account");
    translations["Click to start/stop recording"] = tr("Click to start/stop recording");
    translations["Listening"] = tr("Listening");
    translations["Sleeping"] = tr("Sleeping");
    translations["Microphone not detected"] = tr("Microphone not detected");
    translations["Connection failed, click to try again"] = tr("Connection failed, click to try again");
    translations["Click on the animation or Ctrl+Super+Space to activate"] = tr("Click on the animation or Ctrl+Super+Space to activate");
    translations["Voice input is temporarily unavailable, please check the network!"] = tr("Voice input is temporarily unavailable, please check the network!");
    translations["Unable to connect to the server, please check your network or try again later."] = tr("Unable to connect to the server, please check your network or try again later.");
    translations["Voice conversation"] = tr("Voice conversation");
    translations["Click the animation or press Enter to send"] = tr("Click the animation or press Enter to send");
    translations["Stop recording after %1 seconds"] = tr("Stop recording after %1 seconds");
    translations["Thinking"] = tr("Thinking");
    translations["Click animation to interrupt"] = tr("Click animation to interrupt");
    translations["Answering"] = tr("Answering");
    translations["Your free account quota has been exhausted, please configure your model account to continue using it."] = tr("Your free account quota has been exhausted, please configure your model account to continue using it.");
    translations["Your free account has expired, please configure your model account to continue using it."] = tr("Your free account has expired, please configure your model account to continue using it.");
    translations["UOS AI requires an AI model account to be configured before it can be used. Please configure a model account first."] = tr("UOS AI requires an AI model account to be configured before it can be used. Please configure a model account first.");
    translations["Activate"] = tr("Activate");
    translations["Voice input"] = tr("Voice input");
    translations["Voice broadcast is temporarily unavailable, please check the network!"] = tr("Voice broadcast is temporarily unavailable, please check the network!");
    translations["Turn off voice conversation"] = tr("Turn off voice conversation");
    translations["The picture has been generated, please switch to the chat interface to view it."] = tr("The picture has been generated, please switch to the chat interface to view it.");
    translations["No account, please configure an account"] = tr("No account, please configure an account");
    translations["Answer each question up to 5 times"] = tr("Answer each question up to 5 times");
    translations["Copied successfully"] = tr("Copied successfully");
    translations["Sound output device not detected"] = tr("Sound output device not detected");
    translations["The sound output device is not detected, please check and try again!"] = tr("The sound output device is not detected, please check and try again!");
    translations["Settings"] = tr("Settings");
    translations["About"] = tr("About");
    translations["Mode"] = tr("Mode");
    translations["Window Mode"] = tr("Window Mode");
    translations["Sidebar Mode"] = tr("Sidebar Mode");
    translations["Assistant List"] = tr("Assistant List");
    translations["Agent List"] = tr("Agent List");
    translations["Agent Store"] = tr("Agent Store");
    translations["UOS System Assistant"] = tr("UOS System Assistant");
    translations["Deepin System Assistant"] = tr("Deepin System Assistant");
    translations["Personal Knowledge Assistant"] = tr("Personal Knowledge Assistant");
    translations["Please configure the knowledge base"] = tr("Please configure the knowledge base");
    translations["knowledge base configure content"] = tr("Before using the [Personal Knowledge Assistant], it is necessary to configure the knowledge base. After configuring the knowledge base, AI will answer questions or generate content based on the content you have configured in the knowledge base.");
    translations["Please configure the large model"] = tr("Please configure the large model");
    translations["The personal knowledge assistant can only be used after configuring a large model."] = tr("The personal knowledge assistant can only be used after configuring a large model.");
    translations["To configure"] = tr("To configure");
    translations["To install"] = tr("To install");
    translations["Please install EmbeddingPlugins"] = tr("Please install [EmbeddingPlugins]");
    translations["EmbeddingPlugins install content"] = tr("This assistant requires the installation of the EmbeddingPlugins to run");

    // document summary
    translations["Drag files here to add them."] = tr("Drag files here to add them.");
    translations["You can only add one file, supported file types include: txt, doc, docx, xls, xlsx, ppt, pptx, pdf."] = tr("You can only add one file, supported file types include: txt, doc, docx, xls, xlsx, ppt, pptx, pdf.");
    translations["You can only add a maximum of one file."] = tr("You can only add a maximum of one file.");
    translations["The file format is not supported."] = tr("The file format is not supported.");
    translations["Summarize the key content of the file."] = tr("Summarize the key content of the file.");
    translations["Parsing..."] = tr("Parsing...");
    translations["File Error"] = tr("File Error");
    translations["File has been deleted."] = tr("File has been deleted.");
    translations["The file size exceeds the 100MB limit."] = tr("The file size exceeds the 100MB limit.");
    translations["Upload a document"] = tr("Upload a document");
    translations["File deleted"] = tr("File deleted");
    translations["No text was parsed"] = tr("No text was parsed");
    translations["Reference"] = tr("Reference");

    // Instruction
    translations["Instruction"] = tr("Instruction");
    translations["Type \"/\" in the input box to activate."] = tr("Type \"/\" in the input box to activate.");
    translations["Please enter; “Ctrl+Enter” to change the line."] = tr("Please enter; “Ctrl+Enter” to change the line.");
    translations["Enter your question, or enter \"/\" to select a command\n\"Ctrl+Enter\"  to start a new line"] = tr("Enter your question, or enter \"/\" to select a command\n\"Ctrl+Enter\"  to start a new line");

    //联网搜索
    translations["Search complete."] = tr("Search complete.");
    translations["Click to view results"] = tr("Click to view results");

    // textToPicture
    translations["edit"] = tr("edit");
    translations["save"] = tr("save");
    translations["copy"] = tr("copy");

    // 2.6需求
    translations["Search"] = tr("Search");
    translations["DeepThink(R1)"] = tr("DeepThink(R1)");
    translations["Thinking has stopped"] = tr("Thinking has stopped");
    translations["Back to bottom"] = tr("Back to bottom");
    translations["Deeply thought (%1 seconds)"] = tr("Deeply thought (%1 seconds)");

    // 2.7需求
    translations["New Conversation"] = tr("New Conversation");
    translations["History"] = tr("History");
    translations["No History Records"] = tr("No History Records");
    translations["Today"] = tr("Today");
    translations["Yesterday"] = tr("Yesterday");
    translations["Are you sure to delete the conversation? It will be unrecoverable once deleted."] = tr("Are you sure to delete the conversation? It will be unrecoverable once deleted.");
    translations["The %1 agent used in this conversation has been deleted"] = tr("The %1 agent used in this conversation has been deleted");
    translations["This conversation cannot be viewed. To view it, please install the %1 agent and try again."] = tr("This conversation cannot be viewed. To view it, please install the %1 agent and try again.");
    translations["The original conversation model has been deleted. We have switched to a new model for you to continue the conversation."] = tr("The original conversation model has been deleted. We have switched to a new model for you to continue the conversation.");

    // 2.8需求
    translations["Recommendations"] = tr("Recommendations");
    translations["More"] = tr("More");
    translations["Add Model"] = tr("Add Model");
    translations["No Model"] = tr("No Model");
    translations["No model available. Please install or configure a model in the settings."] = tr("No model available. Please install or configure a model in the settings.");
    translations["Please Describe the Content Theme and Requirements for Your Creation."] = tr("Please Describe the Content Theme and Requirements for Your Creation.");
    translations["Please Enter the Content You Want to Translate and Specify the Target Language. Default Translation is to Chinese."] = tr("Please Enter the Content You Want to Translate and Specify the Target Language. Default Translation is to Chinese.");
    translations["Please Enter the Text You Need to Process and Specify Your Requirements."] = tr("Please Enter the Text You Need to Process and Specify Your Requirements.");
    translations["New Agent Added"] = tr("New Agent Added");
    translations["New Writing, Text Processing, and Translation Agents have been added. Check them out now."] = tr("New Writing, Text Processing, and Translation Agents have been added. Check them out now.");
    translations["Try it"] = tr("Try it");
    translations["Write an article based on the following document:"] = tr("Write an article based on the following document:");
    translations["Translate the following document into English:"] = tr("Translate the following document into English:");

    return translations;
}
